import network_interface
import module
import logging

class Network():
    
    def __init__(self, database, config):
        self.logger = logging.getLogger('hydrobot')
        self.database = database
        self.config = config
        
        self.interfaces = []
        self.load_interfaces()
        self.module_list = module.ModuleList(self, database, config)
        
    def add_interface(self, interface):
        self.interfaces.append(interface)
        
    def load_interfaces(self):
        #load interfaces from config file
        for section in self.config.sections():
            if "interface" in section:
                interface = None
                interface_type = self.config.get(section, "type")
                interface_name = self.config.get(section, "name")
                if interface_type == "CAN":
                    interface = network_interface.CanBusNetworkInterface(self, interface_name)
                    self.add_interface(interface)
                if interface_type == "WIFI":
                    interface = network_interface.CanBusNetworkInterface(self, interface_name)
                    self.add_interface(interface)
                    
    
    def get_interface(self, interface_name):
        for interface in self.interfaces:
            if interface.interface_name == interface_name:
                return interface
    
    def start_interface(self, interface_name):
        self.logger.info("Network: start interface " + interface_name)
        for interface in self.interfaces:
            if interface.interface_name == interface_name:
                interface.start()
    
    def start_all_interfaces(self):
        self.logger.info("Network: start all interfaces")
        for interface in self.interfaces:
            interface.start()
    
    def send_message(self, message):
        module = self.module_list.lookup_module(message.module_uuid)
        if not module == None:
            module.interface.send_message(message)
        else:
            self.logger.warning("Could not find module " + str(message.module_uuid))
    
    def process_message(self, message):
        self.logger.debug("Network: process message")
        self.module_list.lookup_module(message.module_uuid).receive_message(message)
    
